SuperStrict

Framework brl.standardio
Import brl.retro
Import brl.stream
Import brl.pngloader
Import brl.random

Import "pixmapgraphics.bmx"
Import "dictparsenew.bmx"
Import "timestamp.bmx"

Const appname$="Automapper v1.2"

AppTitle=appname

SeedRnd MilliSecs()

' This way you can drag-and-drop a file onto the exe and then it'll be the default path.
Global openwad$=""
If AppArgs.length>1 Then
	openwad=AppArgs[1]
EndIf

' The main loop and stuff
Global c:console=console.readpath("settings.txt")
c.welcome
Repeat
	Print
	c.requestinputs
	Print
	c.do
	Print
	c.finished
Forever

' Thingy for error handling
Function enterexit()
	Print "Press enter to exit.";Input;End
End Function

' Stuff for handling the crappy console UI
Type console
	Global formats$[]=["doom","hexen","zdoom","boom"]
	Field formatname$
	Field inputpath$,outputpath$,width%,height%,margin%,schemename$
	Field mapname$
	Field wstream:TStream,w:wad
	Field schemes:TList=CreateList(),maps:TList
	Field format%,scheme:scheme
	Method finished()
		Print "Finished outputting!"
	End Method
	Method do()
		Local str$="Outputting map"
		If mapname="ALL" Then str:+"s"
		Print str+"..."
		If mapname="ALL" Then
			For Local m$=EachIn maps
				domap(m)
			Next
		Else
			domap(mapname)
		EndIf
		CloseFile wstream
	End Method
	Method domap(name$)
		Local destination$=outputpath
		destination=Replace(destination,"%",name)
		destination=Replace(destination,"*",timestamp.now().todottedstring())
		Local m:map=map.Create(w.getmaplumps(name),format)
		If Not m Then
			Print "Failed to load map lumps for ~q"+name+"~q. Image not outputted."
			Return
		EndIf
		Local pix:TPixmap=m.render(width,height,margin,scheme,w)
		SavePixmapPNG pix,destination
	End Method
	Method welcome()
		Print appname
		Print "Output PNG images of maps in WADs in the same visual style as DOOM's automap."
	End Method
	Method requestinputs()
		Local get$
		Repeat
			If openwad Then
				Print "Opening WAD ~q"+openwad+"~q."
				get=openwad;openwad=""
			Else
				get=getinput("Input WAD path?",inputpath)
			EndIf
			wstream=ReadFile(get)
			If Not wstream
				Print "Error opening WAD file. Try again?"
			Else
				w=wad.read(wstream)
				If Not w
					Print "Error parsing WAD file. Try again?"
				Else
					maps=w.getmapnames()
					If maps.isempty()
						Print "No maps found in WAD. Try again?"
						w=Null
					EndIf
				EndIf
			EndIf
		Until w
		inputpath=get
		Print "Maps in WAD:"
		Print getmapstring(maps)
		Repeat
			Print "Input map? (Input ~qALL~q to output all maps.)"
			Local map$=Input("> ")
			If Upper(map)="ALL" Then
				mapname="ALL"
			Else
				For Local s$=EachIn maps
					If Upper(map)=Upper(s) Then mapname=Upper(s)
				Next
			EndIf
			If Not mapname
				Print "Invalid map name. Try again?"
			EndIf
		Until mapname
		Repeat
			get=getinput("Input WAD format? (Options: Doom, Boom, Hexen, ZDoom) ",formatname)
			format=getformat(get)
			If format=-1 Then
				Print "Invalid map format. Try again?"
			EndIf
		Until format>=0
		formatname=get
		width=Int(getinput("Input automap image width?",width))
		height=Int(getinput("Input automap image height?",height))
		margin=Int(getinput("Input automap image margins?",margin))
		Print "Available color schemes:"
		Print getschemestring()
		Repeat
			get=getinput("Input automap color scheme?",schemename)
			scheme=getscheme(get)
			If Not scheme Then
				Print "Invalid color scheme. Try again?"
			EndIf
		Until scheme
		schemename=get
		outputpath=getinput("Input automap PNG path? (Note: * replaced by date/time. % by map name.)",outputpath)
	End Method
	Function getinput$(valuename$,value$)
		Print valuename
		Print "(Leave blank for ~q"+value+"~q)"
		Local i$=Input("> ")
		If Not i Then i=value
		Return i
	End Function
	Method getschemestring$()
		Local ret$=""
		For Local s:scheme=EachIn schemes
			ret:+s.name+"~n"
		Next
		Return Left(ret,ret.length-1)
	End Method
	Method getmapstring$(maps:TList)
		Local ret$="",i%=0
		For Local s$=EachIn maps
			ret:+" "+s
			i:+1
			If (i Mod 4)=0 Then ret:+"~n"
		Next
		Return Left(ret,ret.length-((i Mod 4)=0))
	End Method
	Method getscheme:scheme(name$)
		name=Lower(name)
		For Local s:scheme=EachIn schemes
			If name=Lower(s.name) Return s
		Next
		Return Null
	End Method
	Function readpath:console(path$)
		Local d:dictnode=dictnode.parsefile(path)
		If Not d Then
			Print "Couldn't parse settings file ~q"+path+"~q."
			enterexit
		EndIf
		Local ret:console=read(d)
		If ret.schemes.isempty() Then
			Print "Settings file ~q"+path+"~q contains no valid color schemes."
			enterexit
		EndIf
		Return ret
	End Function
	Function read:console(d:dictnode)
		Local n:console=New console
		n.formatname=d.getvalue("default_map_format")
		n.width=Int(d.getvalue("default_width"))
		n.height=Int(d.getvalue("default_height"))
		n.margin=Int(d.getvalue("default_margin"))
		n.schemename=d.getvalue("default_scheme")
		n.inputpath=d.getvalue("default_input_path")
		n.outputpath=d.getvalue("default_output_path")
		For Local s:dictnode=EachIn d.getchildren("color_scheme")
			Local ns:scheme=scheme.read(s)
			n.schemes.addfirst ns
		Next
		Return n
	End Function
	Function getformat%(name$)
		name=Lower(name)
		Local ret%=-1
		For Local i%=0 Until formats.length
			If name=Lower(formats[i]) Then ret=i;Exit
		Next
		Return ret
	End Function
End Type

' Stuff for handling color schemes
Type scheme
	Field name$
	Field background%
	Field logic:colorlogic
	Function read:scheme(d:dictnode)
		Local n:scheme=New scheme
		n.name=d.getvalue("scheme_name")
		n.background=Int(d.getvalue("background"))
		n.logic=colorlogic.Create(d.getchild("color_logic"))
		Return n
	End Function
End Type
Type colorlogic
	Field root:colorif
	Function Create:colorlogic(d:dictnode)
		If Not d Return Null
		Local n:colorlogic=New colorlogic
		n.root=colorif.read(d)
		Return n
	End Function
	Method getlineentry:colorif(def:linedef,w:map)
		If Not root Then
			Print "Tried to fetch a line color & layer from a scheme without color_logic."
			enterexit
		EndIf
		Return root.getlineentry(def,w)
	End Method
	Method getmaxlayer%()
		If Not root Return 0
		Return root.getmaxlayer()
	End Method
End Type

Type colorif
	Field conditions:TList=CreateList()
	Field children:TList=CreateList()
	Field color%,layer%
	Method getmaxlayer%()
		Local cmax%=layer
		For Local c:colorif=EachIn children
			cmax=Max(cmax,c.getmaxlayer())
		Next
		Return cmax
	End Method
	Method getlineentry:colorif(def:linedef,w:map)
		For Local c:colorif=EachIn children
			If c.linemeets(def,w) Then
				Return c.getlineentry(def,w)
			EndIf
		Next
		Return Self
	End Method
	Method linemeets%(l:linedef,w:map)
		For Local c$=EachIn conditions
			If Not linemeetscondition(c,l,w) Return 0
		Next
		Return 1
	End Method
	Method linemeetscondition%(condition$,l:linedef,w:map)
		Local notcondition%=0
		If condition[0]=Asc("!") Then notcondition=1;condition=Right(condition,condition.length-1)
		Local ret%=0
		Local flagnames$[]=[	"impassable",		"blockmonster",	"twosided",		"upperunpegged",..
						"lowerunpegged",	"secret",		"blocksound",		"hidden",..
						"shown",		"repeatable",		"",			"",..
						"",			"",			"",			"",..
						"",			"monsteractivates",	"",			"blockeverything"]
		Local specialname$[]=["playerwalk","playeruse","monsterwalk","projectilehit","playerbump","projectilecross","playerusepassthru","projectilehitcross"]
		If condition="exists"
			ret=1		
		ElseIf condition="special"
			ret=l.special>0
		ElseIf condition="tag"
			ret=l.sectortag>0
		ElseIf condition="christmas"
			Local now:timestamp=timestamp.now()
			If now.day=25 And now.month=12 Then ret=1
		ElseIf condition="random"
			ret=Rand(0,1)
		Else
			For Local i%=0 Until flagnames.length
				If condition=flagnames[i] Then
					ret=l.getflag(i)
					Exit
				EndIf
			Next
			If Not ret
				Local trigger%=(l.flags Shr 10) & %111 ''' test if this is right or not
				If condition=specialname[trigger] Then ret=1
			EndIf
			If Not ret
				Local rsec:sector=l.getrightsector(w),lsec:sector=l.getleftsector(w)
				If condition="sectorsecret" Then
					If w.format=wad.DOOM Or w.format=wad.BOOM
						If rsec And rsec.sectortype=9 Then ret=1
						If lsec And lsec.sectortype=9 Then ret=1
						If ret=0 And w.format=wad.BOOM
							If rsec And rsec.sectortype & 128 Then ret=1
							If lsec And lsec.sectortype & 128 Then ret=1
						EndIf
					ElseIf w.format=wad.ZDOOM Or w.format=wad.HEXEN
						If rsec And rsec.sectortype & 1024 Then ret=1
						If lsec And lsec.sectortype & 1024 Then ret=1
					EndIf
				ElseIf condition="sectorhurt" Then ' doesn't work with hexen or zdoom because of format stuff that I can't be bothered to address right now
					If w.format=wad.DOOM Or w.format=wad.BOOM
						If rsec And (rsec.sectortype=4 Or rsec.sectortype=5 Or rsec.sectortype=7 Or rsec.sectortype=11) Then ret=1
						If lsec And (lsec.sectortype=4 Or lsec.sectortype=5 Or lsec.sectortype=7 Or lsec.sectortype=11) Then ret=1
						If ret=0 And w.format=wad.BOOM
							If rsec And rsec.sectortype & 96 Then ret=1
							If lsec And lsec.sectortype & 96 Then ret=1
						EndIf
					EndIf
				ElseIf condition="sectorfloorchange" Then
					If rsec And lsec Then ret=rsec.floorheight<>lsec.floorheight
				ElseIf condition="sectorceilchange" Then
					If rsec And lsec Then ret=rsec.ceilheight<>lsec.ceilheight
				ElseIf condition="sectorlightchange" Then
					If rsec And lsec Then ret=rsec.light<>lsec.light
				ElseIf condition="sectorspecial" Then
					If rsec And rsec.sectortype Then ret=1
					If lsec And lsec.sectortype Then ret=1
				ElseIf condition="sectortag" Then
					If rsec And rsec.sectortag Then ret=1
					If lsec And lsec.sectortag Then ret=1
				EndIf
			EndIf
		EndIf
		If notcondition
			Return Not ret
		Else
			Return ret
		EndIf
	End Method
	Function read:colorif(d:dictnode)
		If Not d Return Null
		Local n:colorif=New colorif
		n.readfrom d
		Return n
	End Function
	Method readfrom(d:dictnode)
		color=Int(d.getvalue("color"))
		layer=Int(d.getvalue("layer"))
		conditions=d.getvalues("condition")
		For Local i%=0 Until 256
			Local child:dictnode=d.getchild("if"+(i+1))
			If Not child Then Exit
			Local n:colorif=colorif.read(child)
			If n
				children.addlast n
			Else
				Print "Something went screwy reading a color scheme in the settings file. Oops."
			EndIf
		Next
	End Method
End Type

' Stuff for processing all that nifty lump data and turning it into usable data
Type map
	Function Create:map(maplumps:lump[],format%=wad.doom)
		If Not maplumps Then Print "no lumps to read map from";Return Null
		Local n:map=New map
		n.format=format
		n.readfrom(maplumps)
		If n.invalidmap Then Print "encountered invalid map";Return Null
		Return n
	End Function
	Field vertexes:vertex[]
	Field linedefs:linedef[]
	Field sidedefs:sidedef[]
	Field sectors:sector[]
	Field invalidmap%=0
	Field format%
	Method readfrom:map(maplumps:lump[])
		Local things_lump:lump	=getlump(maplumps,"THINGS")
		Local linedefs_lump:lump	=getlump(maplumps,"LINEDEFS")
		Local sidedefs_lump:lump	=getlump(maplumps,"SIDEDEFS")
		Local vertexes_lump:lump	=getlump(maplumps,"VERTEXES")
		Local segs_lump:lump	=getlump(maplumps,"SEGS")
		Local ssectors_lump:lump	=getlump(maplumps,"SSECTORS")
		Local nodes_lump:lump	=getlump(maplumps,"NODES")
		Local sectors_lump:lump	=getlump(maplumps,"SECTORS")
		Local reject_lump:lump	=getlump(maplumps,"REJECT")
		Local blockmap_lump:lump	=getlump(maplumps,"BLOCKMAP")
		If Not(things_lump And linedefs_lump And sidedefs_lump And segs_lump And ssectors_lump And vertexes_lump And segs_lump And ssectors_lump And nodes_lump And sectors_lump And reject_lump And blockmap_lump) Then
			invalidmap=1
			Print "failed to read map lumps"
			Return Null
		EndIf
		vertexes=vertex.readlump(vertexes_lump,format)
		linedefs=linedef.readlump(linedefs_lump,format)
		sidedefs=sidedef.readlump(sidedefs_lump,format)
		sectors=sector.readlump(sectors_lump,format)
	End Method
	Method getlump:lump(lumps:lump[],name$)
		For Local l:lump=EachIn lumps
			If l.name=name Or l.trimmedname()=name Then Return l
		Next
		Return Null
	End Method
	
	' This is where the magic takes place
	Method render:TPixmap(xdim%,ydim%,margin%,s:scheme,w:wad)
		Local pix:TPixmap=CreatePixmap(xdim,ydim,PF_RGBA8888)
		ClearPixels pix,s.background
		Local minx%,miny%,maxx%,maxy%
		Local lines:TList[]=getrenderlines(minx,miny,maxx,maxy,s,w)
		For Local i%=1 Until lines.length
			If lines[i] Then
				renderlines(lines[i],pix,minx,miny,maxx,maxy,xdim,ydim,margin)
			EndIf
		Next
		Return pix
	End Method
	Method renderlines(lines:TList,pix:TPixmap,minx%,miny%,maxx%,maxy%,xdim%,ydim%,margin!)
		Local xdiff%=maxx-minx,ydiff%=maxy-miny
		Local xspace%=xdim-margin-margin
		Local yspace%=ydim-margin-margin
		If (xdiff/Double(xspace))>(ydiff/Double(yspace))
			yspace=Double(ydiff)/xdiff*xspace
		Else
			xspace=Double(xdiff)/ydiff*yspace
		EndIf
		Local xmargin!=(xdim-xspace)/2
		Local ymargin!=(ydim-yspace)/2
		For Local line:renderline=EachIn lines
			Local x0!=line.x0
			Local y0!=line.y0
			Local x1!=line.x1
			Local y1!=line.y1
			x0=(x0-minx)/xdiff*xspace+xmargin
			y0=(y0-miny)/ydiff*yspace+ymargin
			x1=(x1-minx)/xdiff*xspace+xmargin
			y1=(y1-miny)/ydiff*yspace+ymargin
			pixline pix,x0,y0,x1,y1,line.color.color
		Next
	End Method
	Const layerlimit%=256
	Method getrenderlines:TList[](minx% Var,miny% Var,maxx% Var,maxy% Var,s:scheme,w:wad)
		minx=0;miny=0;maxx=0;maxy=0
		Local firstminmax%=1
		Local layers%=s.logic.getmaxlayer()
		If layers>layerlimit
			Print "Why does your color scheme need more than "+layerlimit+" layers? That isn't gonna fly."
			enterexit
		EndIf
		Local lines:TList[layers+1]
		For Local i%=0 Until linedefs.length
			Local l:linedef=linedefs[i]
			If l.startvertex<0 Or l.startvertex>=vertexes.length Or l.endvertex<0 Or l.endvertex>=vertexes.length Then
				Print "Encountered invalid vertex ID. Probably using the wrong map format."
				enterexit
			EndIf
			If l.rightside>=sidedefs.length Or l.leftside>=sidedefs.length Then
				Print "Encountered invalid sidedef ID. Probably using the wrong map format."
				enterexit
			EndIf
			Local verts:vertex[]=[vertexes[l.startvertex],vertexes[l.endvertex]]
			Local rside:sidedef,lside:sidedef
			If l.rightside>=0 Then rside=sidedefs[l.rightside]
			If l.leftside>=0 Then lside=sidedefs[l.leftside]
			Local sides:sidedef[]=[rside,lside]
			Local rsec:sector,lsec:sector
			If (rside And (rside.sector<0 Or rside.sector>=sectors.length)) Or (lside And (lside.sector<0 Or lside.sector>=sectors.length)) Then
				Print "Encountered invalid sector ID. Probably using the wrong map format."
				enterexit
			EndIf
			If rside Then rsec=sectors[rside.sector]
			If lside Then lsec=sectors[lside.sector]
			Local sec:sector[]=[rsec,lsec]
			Local impassable%=l.getflag(linedef.impassable)
			Local floorchange%=(sec[0] And sec[1]) And sec[0].floorheight<>sec[1].floorheight
			Local ceilchange%=(sec[0] And sec[1]) And sec[0].ceilheight<>sec[1].ceilheight
			Local col:colorif=s.logic.getlineentry(l,Self)
			Local index%=col.layer
			If Not lines[index] Then lines[index]=CreateList()
			Local line:renderline=New renderline
			line.color=col
			line.x0=verts[0].x
			line.x1=verts[1].x
			line.y0=verts[0].y
			line.y1=verts[1].y
			lines[index].addlast line
			If firstminmax
				firstminmax=0
				minx=Min(line.x0,line.x1)
				miny=Min(line.y0,line.y1)
				maxx=Max(line.x0,line.x1)
				maxy=Max(line.y0,line.y1)
			Else
				minx=Min(Min(minx,line.x0),line.x1)
				miny=Min(Min(miny,line.y0),line.y1)
				maxx=Max(Max(maxx,line.x0),line.x1)
				maxy=Max(Max(maxy,line.y0),line.y1)
			EndIf
		Next
		Return lines
	End Method
	
End Type
Type renderline
	Field x0%,y0%,x1%,y1%,color:colorif
End Type

' Vertexes, duh
Type vertex
	Field x%,y%
	Function read:vertex(f:TStream,format%=wad.DOOM)
		Local n:vertex=New vertex
		n.x=ReadSignedShort(f)
		n.y=-ReadSignedShort(f)
		Return n
	End Function
	Function size%(format%)
		Return 4
	End Function
	Function readlump:vertex[](l:lump,format%=wad.DOOM)
		Local count%=l.size/size(format)
		Local ret:vertex[]=New vertex[count]
		SeekStream l.stream,l.filepos
		For Local i%=0 Until count
			ret[i]=read(l.stream,format)
		Next
		Return ret
	End Function
End Type

' Linedefs, duh again
Type linedef
	Field startvertex@@,endvertex@@
	Field flags@@,special@@,sectortag@@
	Field rightside%,leftside%
	Field args@[5]
	Const impassable%=0,blockmonsters%=1,twosided%=2,upperunpegged%=3,lowerunpegged%=4,secret%=5,blocksound%=6,noautomap%=7,alwaysautomap%=8
	Method getflag%(index%)
		Return flags&(%1 Shl index)
	End Method
	Method getrightside:sidedef(w:map)
		If rightside>=0 And rightside<w.sidedefs.length Then Return w.sidedefs[rightside]
		Return Null
	End Method
	Method getleftside:sidedef(w:map)
		If leftside>=0 And leftside<w.sidedefs.length Then Return w.sidedefs[leftside]
		Return Null
	End Method
	Method getrightsector:sector(w:map)
		Local side:sidedef=getrightside(w)
		If side Return side.getsector(w)
		Return Null
	End Method
	Method getleftsector:sector(w:map)
		Local side:sidedef=getleftside(w)
		If side Return side.getsector(w)
		Return Null
	End Method
	Function read:linedef(f:TStream,format%=wad.DOOM)
		Local n:linedef=New linedef
		n.startvertex=ReadShort(f)
		n.endvertex=ReadShort(f)
		n.flags=ReadShort(f)
		If format=wad.HEXEN Or format=wad.ZDOOM
			n.special=ReadByte(f)
			For Local i%=0 Until n.args.length
				n.args[i]=ReadByte(f)
			Next
		Else
			n.special=ReadShort(f)
			n.sectortag=ReadShort(f)
		EndIf
		n.rightside=ReadSignedShort(f)
		n.leftside=ReadSignedShort(f)
		Return n
	End Function
	Function size%(format%)
		Return 14+((format=wad.HEXEN) Or (format=wad.ZDOOM))*2
	End Function
	Function readlump:linedef[](l:lump,format%=wad.DOOM)
		Local count%=l.size/size(format)
		Local ret:linedef[]=New linedef[count]
		SeekStream l.stream,l.filepos
		For Local i%=0 Until count
			If Eof(l.stream) Then
				Print "Unexpected EOF. Probably using the wrong map format."
				enterexit
			EndIf
			ret[i]=read(l.stream,format)
		Next
		Return ret
	End Function
End Type

' Sidedefs, seriously can't you just read the name of the class
Type sidedef
	Field xoff%,yoff%,uppertex$,lowertex$,middletex$,sector@@
	Method getsector:sector(w:map)
		If sector>=0 And sector<w.sectors.length Then Return w.sectors[sector]
		Return Null
	End Method
	Function read:sidedef(f:TStream,format%=wad.DOOM)
		Local n:sidedef=New sidedef
		n.xoff=ReadSignedShort(f)
		n.yoff=ReadSignedShort(f)
		n.uppertex=ReadString(f,8)
		n.lowertex=ReadString(f,8)
		n.middletex=ReadString(f,8)
		n.sector=ReadShort(f)
		Return n
	End Function
	Function size%(format%)
		Return 30
	End Function
	Function readlump:sidedef[](l:lump,format%=wad.DOOM)
		Local count%=l.size/size(format)
		Local ret:sidedef[]=New sidedef[count]
		SeekStream l.stream,l.filepos
		For Local i%=0 Until count
			ret[i]=read(l.stream,format)
		Next
		Return ret
	End Function
End Type

' Sectors, it's a sector class duh okay
Type sector
	Field floorheight%,ceilheight%,floortex$,ceiltex$
	Field light@@,sectortype@@,sectortag@@
	Function read:sector(f:TStream,format%=wad.DOOM)
		Local n:sector=New sector
		n.floorheight=ReadSignedShort(f)
		n.ceilheight=ReadSignedShort(f)
		n.floortex=ReadString(f,8)
		n.ceiltex=ReadString(f,8)
		n.light=ReadShort(f)
		n.sectortype=ReadShort(f)
		n.sectortag=ReadShort(f)
		Return n
	End Function
	Function size%(format%)
		Return 26
	End Function
	Function readlump:sector[](l:lump,format%=wad.DOOM)
		Local count%=l.size/size(format)
		Local ret:sector[]=New sector[count]
		SeekStream l.stream,l.filepos
		For Local i%=0 Until count
			ret[i]=read(l.stream,format)
		Next
		Return ret
	End Function
End Type

' Because the language's native readshort function doesn't play nice signed numbers, this plays with them for it
Function ReadSignedShort%(f:TStream)
	Local x@@=ReadShort(f)
	If x&$8000 Then
		x=~x
		Return -x-1
	Else
		Return x
	EndIf
End Function

' Class for holding the lumps and stuff
Type wad
	Function read:wad(f:TStream,format%=DOOM)
		Local n:wad=New wad
		n.format=format
		n.readfrom(f)
		If n.invalidwad Return Null
		Return n
	End Function
	Method getmapnames:TList()
		Local ret:TList=CreateList()
		For Local l:lump=EachIn directory
			If Left(l.name,3)="MAP" Then ret.addlast trimmedstring(l.name)
			If l.name[0]=Asc("E") And l.name[1]=Asc("M") Then ret.addlast trimmedstring(l.name)
		Next
		Return ret
	End Method
	Const DOOM%=0,HEXEN%=1,ZDOOM%=2,BOOM%=3 ' formats
	Field format%=DOOM
	Field stream:TStream
	Field identification$,numlumps%,infotableoffset%
	Field directory:lump[]
	Field invalidwad%=0
	Method readfrom(f:TStream)
		stream=f
		SeekStream f,0
		identification=ReadString(f,4)
		If identification<>"PWAD" And identification<>"IWAD" Then invalidwad=1;Return
		numlumps=ReadInt(f)
		infotableoffset=ReadInt(f)
		SeekStream f,infotableoffset
		directory=New lump[numlumps]
		For Local i%=0 Until directory.length
			If Eof(f) Then invalidwad=1;Return
			directory[i]=lump.read(f)
		Next
	End Method
	Const maplumps%=11
	Method getmaplumps:lump[](map$,getlumpcount%=maplumps)
		map=Upper(map)
		For Local i%=0 Until directory.length
			If Upper(directory[i].name)=map Or Upper(trimmedstring(directory[i].name))=map Then
				Local ret:lump[]=New lump[maplumps]
				For Local j%=0 Until maplumps
					ret[j]=directory[i+j]
				Next
				Return ret
			EndIf
		Next
		Return Null
	End Method
End Type

' The lump data
Type lump
	Function read:lump(f:TStream)
		Local n:lump=New lump
		n.readfrom(f)
		Return n
	End Function
	Field stream:TStream
	Field filepos%,size%,name$
	Method readfrom(f:TStream)
		stream=f
		filepos=ReadInt(f)
		size=ReadInt(f)
		name=ReadString(f,8)
	End Method
	Method trimmedname$()
		Return trimmedstring(name)
	End Method
End Type

' Turn DOOM strings with junk null-characters at the end into nice native string objects for easy equivalency comparisons
Function trimmedstring$(name$)
	Local ret$=""
	For Local i%=0 Until name.length
		If name[i]=0 Then Exit
		ret:+Chr(name[i])
	Next
	Return ret
End Function 

